import uuid
import csv
from django.core.management.base import BaseCommand
from django.contrib import admin
from import_export import resources
from import_export.formats.base import BaseFormat
from import_export.widgets import ForeignKeyWidget
from app.models import Subject, Chapter, Question, Choice, ExamCategory


class Command(BaseCommand):
    help = 'Populate fake data from CSV file into the database using Import-Export'

    def handle(self, *args, **kwargs):
        # Open and read the CSV file
        with open('path_to_your_file.csv', mode='r', encoding='utf-8') as f:
            reader = csv.DictReader(f)

            # Create the necessary import resources
            question_resource = QuestionResource()
            choice_resource = ChoiceResource()

            # Prepare a list to insert questions and choices
            questions = []
            choices = []

            for row in reader:
                # Get or create subject and chapter
                subject, created = Subject.objects.get_or_create(name=row['Subject'])
                chapter, created = Chapter.objects.get_or_create(title=row['Chapter'], subject=subject)
                exam_category, created = ExamCategory.objects.get_or_create(name=row['Exam Category'])

                # Prepare the Question object
                question = Question(
                    id=uuid.UUID(row['id']),
                    text=row['Question Text'],
                    question_type=row['Question Type'],
                    subject=subject,
                    chapter=chapter,
                    exam_category=exam_category,
                    difficulty=row['Difficulty']
                )

                questions.append(question)

                # Prepare Choices
                choice_1 = Choice(
                    question=question,
                    text=row['Choice 1'],
                    is_correct=(row['Choice 1'] == row['Correct Choice']),
                    explanation=row['Correct Answer Explanation']
                )
                choice_2 = Choice(
                    question=question,
                    text=row['Choice 2'],
                    is_correct=(row['Choice 2'] == row['Correct Choice']),
                    explanation=row['Correct Answer Explanation']
                )
                choice_3 = Choice(
                    question=question,
                    text=row['Choice 3'],
                    is_correct=(row['Choice 3'] == row['Correct Choice']),
                    explanation=row['Correct Answer Explanation']
                )
                choice_4 = Choice(
                    question=question,
                    text=row['Choice 4'],
                    is_correct=(row['Choice 4'] == row['Correct Choice']),
                    explanation=row['Correct Answer Explanation']
                )

                choices.extend([choice_1, choice_2, choice_3, choice_4])

            # Import the data using bulk_create or ImportExportResource
            # Using bulk_create to avoid multiple database hits
            Question.objects.bulk_create(questions)
            Choice.objects.bulk_create(choices)

            self.stdout.write(self.style.SUCCESS('Data has been successfully populated!'))


# Resources

class QuestionResource(resources.ModelResource):
    question_id = resources.Field(column_name='Question ID', attribute='id')
    question_text = resources.Field(column_name='Question Text', attribute='text')
    question_type = resources.Field(column_name='Question Type', attribute='question_type')
    subject = resources.Field(
        column_name='Subject',
        attribute='subject',
        widget=ForeignKeyWidget(Subject, 'name')
    )
    chapter = resources.Field(
        column_name='Chapter',
        attribute='chapter',
        widget=ForeignKeyWidget(Chapter, 'title')
    )
    exam_category = resources.Field(
        column_name='Exam Category',
        attribute='exam_category',
        widget=ForeignKeyWidget(ExamCategory, 'name')
    )
    difficulty = resources.Field(column_name='Difficulty', attribute='difficulty')

    class Meta:
        model = Question
        fields = ('question_id', 'question_text', 'question_type', 'subject', 'chapter', 'exam_category', 'difficulty')


class ChoiceResource(resources.ModelResource):
    choice_1 = resources.Field(column_name='Choice 1')
    choice_2 = resources.Field(column_name='Choice 2')
    choice_3 = resources.Field(column_name='Choice 3')
    choice_4 = resources.Field(column_name='Choice 4')
    correct_choice = resources.Field(column_name='Correct Choice')
    explanation = resources.Field(column_name='Correct Answer Explanation')

    class Meta:
        model = Choice
        fields = ('choice_1', 'choice_2', 'choice_3', 'choice_4', 'correct_choice', 'explanation')


# Register resources in admin.py for better integration
@admin.register(Question)
class QuestionAdmin(admin.ModelAdmin):
    resource_class = QuestionResource
    list_display = ('text', 'question_type', 'subject', 'chapter', 'exam_category', 'difficulty')
    inlines = [ChoiceInline]


@admin.register(Choice)
class ChoiceAdmin(admin.ModelAdmin):
    resource_class = ChoiceResource
    list_display = ('question', 'text', 'is_correct', 'explanation')


